import JssParser from '../src/JssParser';

/* eslint no-undef: "off" */

describe('JssParser', function () {
  it('parses a simple json css object', function () {
    var parser = new JssParser();
    var res = parser.toCSS({
      '.test': { color: '#fff' },
    });
    expect(res).toBe('.test {\n\tcolor: #fff;\n}\n');
  });

  it('parses a simple comma separated json css object', function () {
    var parser = new JssParser();
    var res = parser.toCSS({
      '.test': { 'top,left,right': 10 },
    });
    expect(res).toBe(
      '.test {\n\ttop: 10px;\n\tleft: 10px;\n\tright: 10px;\n}\n'
    );
  });

  it('parses numbers to pixels', function () {
    var parser = new JssParser();
    var res = parser.toCSS({
      a: { margin: 0 },
    });
    expect(res).toBe('a {\n\tmargin: 0px;\n}\n');
  });

  it('parses array of to pixels', function () {
    // not useful, but correct
    var parser = new JssParser();
    var res = parser.toCSS({
      a: { margin: [0, 10, 0, 10] },
    });
    expect(res).toBe(
      'a {\n\tmargin: 0px;\n\tmargin: 10px;\n\tmargin: 0px;\n\tmargin: 10px;\n}\n'
    );
  });

  it('parses a simple nested json css object', function () {
    var parser = new JssParser();
    var res = parser.toCSS({
      a: { '.selected': { color: 'blue' } },
    });
    expect(res).toBe('a .selected {\n\tcolor: blue;\n}\n');
  });

  it('parses a simple nested json css object', function () {
    var parser = new JssParser();
    var res = parser.toCSS({
      a: { '&.selected': { color: 'blue' } },
    });
    expect(res).toBe('a.selected {\n\tcolor: blue;\n}\n');
  });

  it('parses a simple nested json css object', function () {
    var parser = new JssParser();
    var res = parser.toCSS({
      a: { background_: { color: 'red', image: "url('...')" } },
    });
    expect(res).toBe(
      "a {\n\tbackground-color: red;\n\tbackground-image: url('...');\n}\n"
    );
  });

  it('parses a simple multi valued json css object', function () {
    var parser = new JssParser();
    var res = parser.toCSS({
      div: {
        background_image: [
          '-moz-linear-gradient(white, gray)',
          '-webkit-linear-gradient(white, gray)',
        ],
      },
    });
    expect(res).toBe(
      'div {\n\tbackground-image: -moz-linear-gradient(white, gray);\n\tbackground-image: -webkit-linear-gradient(white, gray);\n}\n'
    );
  });

  it('parses a simple @font-face object', function () {
    var parser = new JssParser();
    var res = parser.toCSS({
      '@font-face': { 'font-family': 'MyFont', src: "url('//...')" },
    });

    expect(res).toBe(
      "@font-face {\n\tfont-family: MyFont;\n\tsrc: url('//...');\n}\n"
    );
  });

  it('parses a multiple @font-face objects', function () {
    var parser = new JssParser();
    var res = parser.toCSS({
      '@font-face': [
        { 'font-family': 'MyFont', src: "url('//...')" },
        { 'font-size': 10, 'font-weight': '100' },
      ],
    });

    expect(res).toBe(
      "@font-face {\n\tfont-family: MyFont;\n\tsrc: url('//...');\n}\n@font-face {\n\tfont-size: 10px;\n\tfont-weight: 100;\n}\n"
    );
  });

  it('parses a multiple @font-face objects with multiple sources', function () {
    var parser = new JssParser();
    var res = parser.toCSS({
      '@font-face': [
        { 'font-family': 'MyFont', src: "url('//...')" },
        {
          'font-size': 10,
          'font-weight': '100',
          src: ["url('//one')", "url('//two')"],
        },
      ],
    });

    expect(res).toBe(
      "@font-face {\n\tfont-family: MyFont;\n\tsrc: url('//...');\n}\n@font-face {\n\tfont-size: 10px;\n\tfont-weight: 100;\n\tsrc: url('//one');\n\tsrc: url('//two');\n}\n"
    );
  });

  it('parses colors', function () {
    var parser = new JssParser();
    var res = parser.toHex(255, 255, 255);

    expect(res).toBe('#ffffff');
  });

  it('parses colors', function () {
    var parser = new JssParser();
    var res = parser.toHex(1, 2, 3);

    expect(res).toBe('#010203');
  });

  it('parses factors colors', function () {
    var parser = new JssParser();
    var res = parser.color('#45c301');

    expect(res).toBe('#45c301');
  });

  it('parses factors colors', function () {
    var parser = new JssParser();
    var res = parser.color('#45c301', 0);

    expect(res).toBe('#000000');
  });

  it('parses factors colors', function () {
    var parser = new JssParser();
    var res = parser.color('#0f0d04', 0.5);

    expect(res).toBe('#070602');
  });
});
