/*
 * Handles communication between windows
 */
import objectUtils from './utils/objectUtils';
import stringUtils from './utils/stringUtils';
import startsWith from 'babel-runtime/core-js/string/starts-with';

export default class Messenger {
  /*
   * _window is the window to listen on.
   * targetWindow is the window to communicate with.
   * targetOorigin is the origin url of the target.
   */
  constructor(_window, targetWindow, targetOrigin) {
    this.window = _window || window;
    this._onHandleMessage = this._onHandleMessage.bind(this);

    this.targetOrigin = targetOrigin || '*';
    this.onMessengerHandler = null;
    this.targetWindow = targetWindow || null;
    this.userData = null;
    this.prefix = '';
    this.nextId = 0;
    this.name = 'messenger';
  }

  /*
   * Handles a message form another window.
   */
  _onHandleMessage(event) {
    if (
      objectUtils.isString(event.data) == false ||
      (this.prefix && !startsWith(event.data, this.prefix))
    ) {
      return;
    }

    var json = this.prefix
      ? event.data.substring(this.prefix.length)
      : event.data;

    //console.log('Message Received ' + this.name + ': ' + window.location + ' raw json: ' + json);

    var data = JSON.parse(json);

    if (data.type) {
      var name = 'on' + stringUtils.toCamelCase(data.type, '_', '', true);
      if (
        this.onMessengerHandler &&
        name in this.onMessengerHandler &&
        typeof this.onMessengerHandler[name] === 'function'
      ) {
        this.onMessengerHandler[name](this, event, data);
      }
    }
  }

  _postMessage(message) {
    if (
      this.targetWindow &&
      this.targetWindow.postMessage &&
      this.targetOrigin
    ) {
      var messageString = this.prefix + JSON.stringify(message);
      //console.log('Message sent ' + this.name + ': ' + window.location + ' origin: ' + this.targetOrigin + ' message: ' + messageString);

      this.targetWindow.postMessage(messageString, this.targetOrigin);
    }
  }

  postMessage(type, data) {
    var message = objectUtils.extend({}, data);
    message.type = type;
    message.messageid = (this.name || 'messenger') + '-' + this.nextId++;
    this._postMessage(message);
  }

  setName(name) {
    this.name = name;
  }

  /*
   * The prefix allows a tag to be added as a prefix to a sent message. Upon
   * receiving the message the message is checked to ensure the message starts
   * with prefix. Prefix is stripped before parsing the message a json. This
   * ensures that we only parse the proper messages.
   */
  setPrefix(prefix) {
    this.prefix = prefix;
  }

  setUserData(udata) {
    this.userData = udata;
  }

  getUserData() {
    return this.userData;
  }

  startListening() {
    this.window.addEventListener('message', this._onHandleMessage, false);
  }

  stopListening() {
    this.window.removeEventListener('message', this._onHandleMessage, false);
  }

  setTargetOrigin(url) {
    this.targetOrigin = url;
  }
}
