class DocumentReady {
  constructor() {
    this._readyList = [];
    this._readyFired = false;
    this._readyEventHandlersInstalled = false;
    this._onReady = this._onReady.bind(this);
    this._onReadyStateChange = this._onReadyStateChange.bind(this);
  }

  // call this when the document is ready
  // this function protects itself against being called more than once
  _onReady() {
    if (!this._readyFired) {
      // this must be set to true before we start calling callbacks
      this._readyFired = true;
      for (var i = 0; i < this._readyList.length; i++) {
        // if a callback here happens to add new ready handlers,
        // the docReady() function will see that it already fired
        // and will schedule the callback to run right after
        // this event loop finishes so all handlers will still execute
        // in order and no new ones will be added to the readyList
        // while we are processing the list
        this._readyList[i].fn(this._readyList[i].ctx);
      }
      // allow any closures held by these functions to free
      this._readyList = [];

      if (this._readyEventHandlersInstalled) {
        if (document.removeEventListener) {
          document.removeEventListener('DOMContentLoaded', this._onReady);
          window.removeEventListener('load', this._onReady);
        } else {
          // must be IE
          document.detachEvent('onreadystatechange', this._onReadyStateChange);
          window.detachEvent('onload', this._onReady);
        }
        this._readyEventHandlersInstalled = false;
      }
    }
  }

  _onReadyStateChange() {
    if (document.readyState === 'complete') {
      this._onReady();
    }
  }

  // This is the one public interface
  // docReady(fn, context);
  // the context argument is optional - if present, it will be passed
  // as an argument to the callback
  ready(callback, context) {
    // if ready has already fired, then just schedule the callback
    // to fire asynchronously, but right away
    if (this._readyFired) {
      setTimeout(function () {
        callback(context);
      }, 1);
      return;
    } else {
      // add the function and context to the list
      this._readyList.push({
        fn: callback,
        ctx: context,
      });
    }
    // if document already ready to go, schedule the ready function to run
    if (document.readyState === 'complete') {
      setTimeout(this._onReady, 1);
    } else if (!this._readyEventHandlersInstalled) {
      // otherwise if we don't have event handlers installed, install them
      if (document.addEventListener) {
        // first choice is DOMContentLoaded event
        document.addEventListener('DOMContentLoaded', this._onReady);
        // backup is window load event
        window.addEventListener('load', this._onReady);
      } else {
        // must be IE
        document.attachEvent('onreadystatechange', this._onReadyStateChange);
        window.attachEvent('onload', this._onReady);
      }
      this._readyEventHandlersInstalled = true;
    }
  }
}

export default new DocumentReady();
